// The GranFileInstrument uses GranulateRandom to chop up an audio file
//   into sound grains which are random in length and spaced randomly within
//   the parameters given.

// Every instrument must implement the Instrument interface so 
// playNote() can call the instrument's noteOn() and noteOff() methods.
class GranFileInstrument implements Instrument
{
  // Create all variables that must be used throughout the class
  ADSR adsr;        // Set in noteOn() and noteOff()
  Pan pan;          // Set in noteOn() and noteoff()
  Sampler snd;
  MultiChannelBuffer sampleBuffer = new MultiChannelBuffer( 1, 1024 );

  // Constructor for this instrument.  The file name and amplitude
  // are passed in.
  GranFileInstrument(String fileName, float amplitude)
  {
    // Load the file passed into the constructor into a Buffer
    // and remember the sample rate.
    float sampleRate = minim.loadFileIntoBuffer(fileName, sampleBuffer);
    
    // If the file loaded, there will be a non-zero sample rate
    if (sampleRate > 0)
    {
      // Reverse the buffer half the time
      if (random(10) < 5)
        reverseBuff(sampleBuffer);

      // Create a sampler that will use our buffer to generate audio.
      // We must provide the sample rate of the audio and the number of voices. 
      snd = new Sampler(sampleBuffer, sampleRate, 1);
      //snd = new Sampler (fileName, 1, minim);  // Read audio file to granulate

      // Granulate the sound file.  Parameters are:
      // grainLengMin, spaceLengMin, fadeLengMin, grainLengMax, spaceLengMax, fadeLengMax
      GranulateRandom chopper =
        new GranulateRandom(0.010, 0.005, 0.005, 0.050, 0.020, 0.010);

      // Envelope for the overall cloud to avoid clicks at the end
      // Parameters are: maxAmp, attTime, decTime, susLvl, relTime
      adsr = new ADSR(amplitude, 0.02, 0.02, amplitude, 0.05);

      // A pan control controlled by a randomly set LFO
      pan = new Pan(0);
      // Low frequency oscillator to control pan location
      Oscil LFO = new Oscil(random (0.05, 0.5), 1.0, Waves.TRIANGLE);
      LFO.patch(pan.pan);

      // Patch everything together up until the output
      // The sound file goes into the chopper
      snd.patch(chopper);
      // The chopper goes into the envelope
      chopper.patch(adsr);
      // Envelope goes into the pan to sweep it across stereo field randomly
      adsr.patch(pan);
    }
  }

  // Every instrument must have a noteOn( float ) method
  void noteOn( float dur )
  {
    // Turn on the envelope
    adsr.noteOn();
    // Trigger the sound file to start chain
    snd.trigger();
    // and patch end of the chain to the output
    pan.patch(out);
  }

  // Every instrument must have a noteOff() method 
  void noteOff()
  {
    // Tell the ADSR to unpatch after the release is finished
    adsr.unpatchAfterRelease(out);
    // Call the envelope noteOff() to shut it down
    adsr.noteOff();
  }
}

// Reverses the order of the samples in the Buffer passed into the function
void reverseBuff (MultiChannelBuffer sampleBuffer)
{
  // Find the middle of the buffer
  int buffLeng = sampleBuffer.getBufferSize(); 
  int halfLeng = sampleBuffer.getBufferSize() / 2; 

  // Reverse the sample buffer to play it backwards by exchanging samples
  // equidistant from the edges
  // Go halfway into the buffer
  float temp;      // Declare outside the loop
  int otherSideS;
  for ( int s = 0; s < halfLeng; ++s )
  {
    temp = sampleBuffer.getSample( 0, s );   // Copy the left sample
    otherSideS = buffLeng - s - 1;           // Get right sample location
    // Copy right sample over left sample
    sampleBuffer.setSample(0, s, sampleBuffer.getSample(0, otherSideS));
    // Copy saved left sample over right sample to complete the exchange
    sampleBuffer.setSample(0, otherSideS, temp);
  }
}