/**
 * This sketch demonstrates how to patch an audio file into
 * a ring modulator.  The audio file (the carrier) needs to be played
 * by a Sampler ugen object so that it can be patched with other ugens.
 * Actually, the Sampler class is probably better for sound effects
 * than the SamplePlayer class anyway because it is more flexible.
 */

import ddf.minim.*;
import ddf.minim.ugens.*;
import ddf.minim.analysis.*;

Minim minim;
AudioOutput out;    // Output to the sound card
FFT fft;            // Used for spectral display
PFont legendFont;
PFont scaleFont;

RingFile rF1;       // Declare ring modulator from RingFile class

void setup()
{
  size(1024, 400);
  legendFont = createFont("Helvetica", 16); // This works for standard fonts
  scaleFont = createFont("Helvetica", 10);  // This works for standard fonts
  minim = new Minim(this);

  // use the getLineOut method of the Minim object to get an AudioOutput object
  out = minim.getLineOut();

  // Set up the FFT for the display
  fft = new FFT( out.bufferSize(), out.sampleRate() );
  
  // Set up the ring modulator with the file as carrier and intial values
  // for modulating frequency and amplitude.
  rF1 = new RingFile("Home.wav", 500, 0.5);
  println("Trigger sound file with 't'");
}

// Display stuff for the audio output.  None of the code in draw() has
// anything to do with how the ring modulator functions.  It just displays
// what comes over the audio output.
void draw()
{
  background(0);                    // White on black display
  stroke(255);

  textFont(legendFont);             // Set up and print the legend and param values
  fill(255);
  text("Mod Freq: " + rF1.modF, width-185, 20);
  text("Mod Amp: " + rF1.modAmp, width-185, 40);
  text("Trigger sound file with ' t '", width-185, 60);

  // Draw the waveforms (just need one channel)
  stroke(0, 255, 0);
  for (int i = 0; i < out.bufferSize () - 1; i++)
  {
    line( i, 100 + out.left.get(i)*100, i+1, 100 + out.left.get(i+1)*100 );
    //line( i, 150 + out.right.get(i)*50, i+1, 150 + out.right.get(i+1)*50 );
  }

  // Draw the real-time spectrum
  fft.forward(out.left);            // Adapted from the FFT example program
  textFont(scaleFont);              // Set up the font for the scale
  stroke(255);
  for (int i = 0; i < fft.specSize (); i++)
  {
    // Draw the line for frequency band i, scaling it by 4 so we can see it better
    line(i*2, height*0.85, i*2, height*0.85 - fft.getBand(i) * 3);
    if (i%16 == 0)
    {
      if (i%32 == 0)
      {
        line (i*2, height*0.85, i*2, height*0.85 + 10);
        text(int(fft.indexToFreq(i)), i*2, height*0.85+19);
      } else
      {
        line (i*2, height*0.85, i*2, height*0.90 + 10);
        text(int(fft.indexToFreq(i)), i*2, height*0.90+19);
      }
    }
  }
}

// Change the parameters of the ring modulation Oscil in real-time
// using the mouse.  X dimension controls modulation frequency.
// Y dimension controls modulation amplitude (depth).
void mouseMoved()
{
  float modF = map( mouseX, 0, width, 0.1, 6000 );  // Range 0.1 to 6000 Hz
  float modAmp = map( mouseY, 0, height, 4, 0.01 ); // Range 0.01 to 4

  rF1.setModF(modF);      // Call methods in the class to set values
  rF1.setModAmp(modAmp);
}

void keyPressed()
{
  if (key == 't')
    rF1.trigger();        // Trigger the recorded sound
}
