/* Additive Instrument class
 * Implements a 3-voice additive instrument with a fundamental
 * and 2 harmonics above that.  The harmonics are specified as
 * halfsteps above the fundamental.  This instrument will be configured
 * and will start playing when it is instantiated in setup().
 */

class AddInst
{
  Oscil wave1;   // 3 waveforms, all sines in this case
  Oscil wave2;
  Oscil wave3;
  Summer sum;    // The additive thing
  float fund;    // Fundamental frequency
  int harm1;     // Offset of 1st harmonic in eTR
  int harm2;     // Offset of 2nd harmonic in eTR
  float amp;     // Amplitude of each voice

  // Equal Temperment Ratios for successive halfsteps above the fundamental
  // Used to get "musical" frequencies for upper harmonics
  float [] eTR =
    {
    1.0, 1.05946, 1.12246, 1.18921, 1.25992, 1.33483, 1.41421, 
    1.49831, 1.58740, 1.68179, 1.78180, 1.88775, 2.0
  };

  // Additive instrument with fundamental frequency f and
  // harmonics h1 and h2 half steps above that up to 12.
  AddInst(float f, int h1, int h2)
  {
    fund = f;
    harm1 = h1;
    harm2 = h2;
    amp = 0.3f;    // Only 3 voices so sum is 0.9 => won't clip
    sum = new Summer();

    wave1 = new Oscil(fund, amp, Waves.SQUARE );
    wave1.patch( sum );        // patch the Oscil to the Summer

    wave2 = new Oscil(fund*eTR[harm1], amp, Waves.SQUARE );
    wave2.patch( sum );

    wave3 = new Oscil(fund*eTR[harm2], amp, Waves.SQUARE );
    wave3.patch( sum );

    // Patch the Summer to the output, and you should hear a chord
    sum.patch( out );
  }

  // (Re)set frequencies of the 3 Oscils to fundamental (f)
  // and 2 harmonics above it.  Called in mouseMoved().
  void setF(float f)
  {
    fund = f;
    wave1.frequency.setLastValue( fund );
    wave2.frequency.setLastValue( fund * eTR[harm1] );
    wave3.frequency.setLastValue( fund * eTR[harm2] );
  }

  // Set amplitudes of all three Oscils to parameter a.
  // Called in mouseMoved().
  void setAmp(float a)
  {
    amp = a;
    adIn1.wave1.amplitude.setLastValue( amp );
    adIn1.wave2.amplitude.setLastValue( amp );
    adIn1.wave3.amplitude.setLastValue( amp );
  }

  // Sets harmonics in the 3-note chord
  // Called in keyPressed()
  void setHarms(char chord)
  {
    switch (chord)
    {
    case 'M':
      harm1 = 4;    // Major triad
      harm2 = 7;
      break;
    case 'm':
      harm1 = 3;    // Minor triad
      harm2 = 7;
      break;
    case 'd':
      harm1 = 3;    // Diminished triad
      harm2 = 6;
      break;
    case 'a':
      harm1 = 4;    // Augmented triad
      harm2 = 8;
      break;
    case 'f':
      harm1 = 5;    // Ascending 4ths
      harm2 = 10;
      break;
    case 's':
      harm1 = 1;    // Minor 2nds
      harm2 = 2;
      break;
    default:
      break;
    }
  }

  // Sets waveform in all 3 waves
  // Called in keyPressed()
  void setWave(char waveForm)
  {
    switch( waveForm )
    {
    case '1': 
      wave1.setWaveform( Waves.SINE );
      wave2.setWaveform( Waves.SINE );
      wave3.setWaveform( Waves.SINE );
      break;
    case '2':
      wave1.setWaveform( Waves.TRIANGLE );
      wave2.setWaveform( Waves.TRIANGLE );
      wave3.setWaveform( Waves.TRIANGLE );
      break;
    case '3':
      wave1.setWaveform( Waves.SAW );
      wave2.setWaveform( Waves.SAW );
      wave3.setWaveform( Waves.SAW );
      break;
    case '4':
      wave1.setWaveform( Waves.SQUARE );
      wave2.setWaveform( Waves.SQUARE );
      wave3.setWaveform( Waves.SQUARE );
      break;
    case '5':
      wave1.setWaveform( Waves.QUARTERPULSE );
      wave2.setWaveform( Waves.QUARTERPULSE );
      wave3.setWaveform( Waves.QUARTERPULSE );
      break;
    case '6':
      wave1.setWaveform( Waves.PHASOR );
      wave2.setWaveform( Waves.PHASOR );
      wave3.setWaveform( Waves.PHASOR );
      break;
    default: 
      break;
    }
  }
}
