/**
 * This sketch demonstrates how to patch more than one UGen
 * to a Summer. It works just like patching any other two 
 * UGens together, the difference being that a Summer 
 * can have more than one UGen patched to it and will 
 * output the sum of all signals, whereas most other UGens
 * allow only one input.
 *
 * Mouse controls frequency (x dimension) & amplitude (y dim)
 * 
 * This sketch also uses an Additive Instrument class to create
 * an object that will play the sounds.  This instrument class
 * encapsulates all the details needed to make non-trivial sounds
 * that can be altered in real time.
 *
 * For many more examples of UGens included with Minim, 
 * have a look in the Synthesis folder of the Minim examples.
 * 
 * For more information about Minim and additional features, 
 * visit http://code.compartmental.net/minim/
 */

import ddf.minim.*;
import ddf.minim.ugens.*;
import ddf.minim.analysis.*;  // Needed for fft object to do spectrum

Minim minim;
AudioOutput out;    // Global output channel
AddInst adIn1;      // Additive instrument to play
float fund;         // Fundamental to scale harmonics from
FFT fft;            // Used to display the real-time spectrum

void setup()
{
  size(512, 200);

  // Set up a Minim object to do all the audio stuff
  minim = new Minim(this);

  // Use getLineOut method of the Minim object to get
  // an AudioOutput object
  out = minim.getLineOut();

  // Set up an additive instrument to start playing a chord
  fund = 2000;
  // Uncomment one of the following (Now, can set with key hits
  adIn1 = new AddInst(fund, 4, 7);  // Major triad (C E G)
  //adIn1 = new AddInst(fund, 3, 7);  // Minor triad C Eb G)
  //adIn1 = new AddInst(fund, 3, 6);  // Diminished triad C Eb Gb)
  //adIn1 = new AddInst(fund, 4, 8);  // Augmented triad C E G#)
  //adIn1 = new AddInst(fund, 5, 10); // Ascending 4ths (C F Bb)
  //adIn1 = new AddInst(fund, 1, 2);  // Minor 2nds (C C# D: very dissonent)

  // Set up the FFT for the spectrum display of the output signal
  fft = new FFT( out.bufferSize(), out.sampleRate() );
  
  // Print UI keys on the Console window
  print("M: Major, m: minor, d: diminished, a: augmented, ");
  println(" f: 4ths, s: minor 2nds");
  print("1: Sine, 2: Triangle, 3: Saw, 4: Square, ");
  println("5: Quarter Pulse, 6: Phasor");
}

void draw()
{
  background(0);
  // Draw the waveform shape we are using in the oscillator
  stroke( 0, 255, 0 );  // Green
  strokeWeight(4);      // Big pixels
  for( int i = 0; i < width-1; ++i )
  {
    point(i,
      height/2 - (height*0.49)*adIn1.wave1.getWaveform().value((float)i/width));
  }

  stroke(255);
  strokeWeight(3);

  // Draw the waveform of the output near top of window
  for (int i = 0; i < out.bufferSize() - 1; i++)
  {
    line( i, 50  - out.left.get(i)*50, i+1, 50  - out.left.get(i+1)*50);
    //line( i, 150 - out.right.get(i)*50, i+1, 150 - out.right.get(i+1)*50);
  }

  // Get the spectrum and display it growing out of bottom of window
  fft.forward(out.left);
  stroke(255);
  for (int i = 0; i < fft.specSize (); i++)
  {
    // Draw the line for frequency band i, scaling it by factor of 6
    // to spread out the audible frequencies on the x axis
    line(i*6, height, i*6, height - fft.getBand(i));
  }
}

// Alter parameters for adIn1: x maps to frequency, y maps to amplitude
void mouseMoved()
{
  float amp = map( mouseY, 0, height, 0.3, 0.0 );  // 0.3 is max amplitude
  float f = map( mouseX, 0, width, 1.0, fund );

  adIn1.setF(f);
  adIn1.setAmp(amp);
}

// Alter waveform in all 3 chord voices or change chord's harmonics
void keyPressed()
{
  if (key >= '1' && key <= '6')
    adIn1.setWave(key);          // Change waveforms
  else {
    adIn1.setHarms(key);
    float f = map( mouseX, 0, width, 1.0, fund );
    adIn1.setF(f);               // Change harmonics in chord
  }
}
